/* CBT Exam — HTML/CSS/JS + Web Basics
   Objective: 40 (auto-mark)
   Theory/Practical: 4 (manual scoring)

   + Hosted mode with PHP + MySQL (AJAX)
   + Real-time save (MCQ + theory)
   + Violation tracking (tab switch / blur / exit fullscreen)
*/

let attemptToken = null;

async function apiPost(url, data) {
    const res = await fetch(url, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(data)
    });
    // if server returns non-json, this will throw
    return res.json();
}

const EXAM_SECONDS = 90 * 60; // 1 hour 30 mins

// ========== SETTINGS ==========
const MAX_VIOLATIONS = 3;         // auto-submit after this
const THEORY_AUTOSAVE_MS = 5000;  // autosave theory every 5s
const HEARTBEAT_MS = 10000;       // ping server every 10s

// 40 MCQ questions. correct = index (0..3)
const mcqs = [
    // HTML (10)
    { q: "What does HTML stand for?", a: ["Hyperlinks and Text Markup Language", "HyperText Markup Language", "Home Tool Markup Language", "Hyper Transfer Markup Language"], correct: 1 },
    { q: "Which tag is used to create a hyperlink?", a: ["<link>", "<a>", "<href>", "<url>"], correct: 1 },
    { q: "Which attribute is used to provide an image source?", a: ["href", "src", "alt", "link"], correct: 1 },
    { q: "Which tag is best for the main heading of a page?", a: ["<h6>", "<head>", "<h1>", "<title>"], correct: 2 },
    { q: "Which element is used to create an ordered list?", a: ["<ul>", "<ol>", "<li>", "<dl>"], correct: 1 },
    { q: "Which tag is used for a line break?", a: ["<br>", "<hr>", "<break>", "<lb>"], correct: 0 },
    { q: "Which is a semantic element?", a: ["<div>", "<span>", "<section>", "<b>"], correct: 2 },
    { q: "Which attribute makes an input mandatory?", a: ["must", "required", "validate", "needed"], correct: 1 },
    { q: "Which tag holds metadata and links to CSS?", a: ["<body>", "<main>", "<head>", "<footer>"], correct: 2 },
    { q: "What is the correct doctype for HTML5?", a: ["<!DOCTYPE html>", "<!DOCTYPE HTML5>", "<!DOCTYPE web>", "<!DOC html>"], correct: 0 },

    // CSS (10)
    { q: "CSS stands for:", a: ["Creative Style Sheets", "Cascading Style Sheets", "Computer Style System", "Colorful Style Sheets"], correct: 1 },
    { q: "Which CSS property changes text color?", a: ["font-color", "text-color", "color", "fgcolor"], correct: 2 },
    { q: "Which selector targets an element with id='box'?", a: [".box", "#box", "box", "*box"], correct: 1 },
    { q: "Which property controls spacing inside the border?", a: ["margin", "padding", "gap", "outline"], correct: 1 },
    { q: "Which property controls spacing outside the border?", a: ["margin", "padding", "spacing", "border-gap"], correct: 0 },
    { q: "Which layout uses rows/columns explicitly?", a: ["Float", "Flexbox", "Grid", "Position"], correct: 2 },
    { q: "Which unit is relative to the root font size?", a: ["em", "rem", "px", "vh"], correct: 1 },
    { q: "To make text bold in CSS you can use:", a: ["font-style: bold;", "font-weight: bold;", "text-weight: bold;", "font-bold: true;"], correct: 1 },
    { q: "Which property makes corners rounded?", a: ["corner-radius", "border-round", "border-radius", "radius"], correct: 2 },
    { q: "Media queries are used for:", a: ["Animations", "Responsive design", "Database queries", "Linking JS"], correct: 1 },

    // JavaScript (12)
    { q: "JavaScript is mainly used to:", a: ["Style pages", "Structure pages", "Add interactivity", "Store images"], correct: 2 },
    { q: "Which keyword declares a block-scoped variable?", a: ["var", "let", "define", "int"], correct: 1 },
    { q: "Which is a correct function syntax?", a: ["function myFn() {}", "func myFn() {}", "def myFn() {}", "fn myFn() {}"], correct: 0 },
    { q: "What does DOM stand for?", a: ["Document Object Model", "Data Object Method", "Document Oriented Mode", "Display Object Management"], correct: 0 },
    { q: "How do you select an element by id in JS?", a: ["document.query('id')", "document.getElementById('id')", "document.getById('id')", "window.getElementById('id')"], correct: 1 },
    { q: "Which operator checks strict equality?", a: ["=", "==", "===", "!="], correct: 2 },
    { q: "What is the output type of prompt()?", a: ["Number", "String (or null)", "Boolean", "Object"], correct: 1 },
    { q: "Which converts '12' to number safely?", a: ["Number('12')", "toNumber('12')", "int('12')", "parse('12')"], correct: 0 },
    { q: "Which is NOT a JS data type?", a: ["String", "Boolean", "Float", "Undefined"], correct: 2 },
    { q: "Which method adds an item to end of array?", a: ["push()", "pop()", "shift()", "unshift()"], correct: 0 },
    { q: "Event listener syntax is:", a: ["element.on('click', fn)", "element.addEventListener('click', fn)", "element.listen('click', fn)", "element.click(fn)"], correct: 1 },
    { q: "JSON stands for:", a: ["Java Source Object Notation", "JavaScript Object Notation", "Joined String Object Notation", "JavaScript Ordered Name"], correct: 1 },

    // WordPress + Web basics (8)
    { q: "WordPress is mainly a:", a: ["Programming language", "CMS (Content Management System)", "Database server", "Browser"], correct: 1 },
    { q: "In WordPress, a Theme controls:", a: ["Site appearance/layout", "Database backup", "Email sending", "DNS records"], correct: 0 },
    { q: "A plugin in WordPress is used to:", a: ["Change domain name", "Extend features/functionality", "Replace hosting", "Edit DNS"], correct: 1 },
    { q: "IP address is used to:", a: ["Style websites", "Identify a device on a network", "Encrypt CSS", "Create domains"], correct: 1 },
    { q: "DNS is best described as:", a: ["A programming language", "A system that translates domain names to IP addresses", "A hosting company", "A browser extension"], correct: 1 },
    { q: "A domain name is:", a: ["A website color scheme", "Human-friendly name for an IP (e.g., example.com)", "A file folder", "A CSS property"], correct: 1 },
    { q: "HTTP vs HTTPS: HTTPS is:", a: ["Slower always", "Encrypted/secure version of HTTP", "A different browser", "Only for emails"], correct: 1 },
    { q: "The internet is:", a: ["One big computer", "A global network of networks", "A single website", "A database tool"], correct: 1 },
];

// 4 theory/coding practical prompts
const theory = [
    {
        title: "Q1 (Theory) — Internet Basics",
        prompt: `Explain the difference between:
1) IP address and Domain name
2) Internet and World Wide Web (WWW)
Give one example for each.`,
        hint: "Write in your own words with examples."
    },
    {
        title: "Q2 (Theory) — DNS Flow",
        prompt: `Describe step-by-step what happens when a user types 'google.com' in the browser and presses Enter.
Mention DNS, IP, HTTP/HTTPS, and server response.`,
        hint: "Bullet points are fine."
    },
    {
        title: "Q3 (Coding Practical) — HTML/CSS Layout",
        prompt: `Build a simple responsive card using HTML + CSS:
- A card with an image area (use a colored div), a title, and a button
- On desktop: card width 400px centered
- On mobile (<600px): card width 100%
Provide BOTH HTML and CSS code.`,
        hint: "Use media query."
    },
    {
        title: "Q4 (Coding Practical) — JavaScript Logic",
        prompt: `Write JavaScript code that:
- Asks for a student's score (0–100)
- Displays grade:
  70–100: A
  60–69: B
  50–59: C
  45–49: D
  0–44: F
- If input is invalid, show an error message.
Provide the code.`,
        hint: "Use prompt(), Number(), if/else."
    },
];

// ---------- DOM ----------
const startBtn = document.getElementById("startBtn");
const examArea = document.getElementById("examArea");
const resultArea = document.getElementById("resultArea");
const mcqWrap = document.getElementById("mcqWrap");
const theoryWrap = document.getElementById("theoryWrap");
const timerEl = document.getElementById("timer");
const liveScoreEl = document.getElementById("liveScore");

const submitBtn = document.getElementById("submitBtn");
const submitBtn2 = document.getElementById("submitBtn2");

const rName = document.getElementById("rName");
const rId = document.getElementById("rId");
const rObj = document.getElementById("rObj");
const manualScore = document.getElementById("manualScore");
const rTotal = document.getElementById("rTotal");
const reviewArea = document.getElementById("reviewArea");
const toggleReviewBtn = document.getElementById("toggleReviewBtn");
const printBtn = document.getElementById("printBtn");
const resetBtn = document.getElementById("resetBtn");

let timer = null;
let remaining = EXAM_SECONDS;
let started = false;
let submitted = false;

// anti-cheat
let violations = 0;
let theoryAutosaveTimer = null;
let heartbeatTimer = null;

function pad(n) { return String(n).padStart(2, "0"); }
function formatTime(sec) {
    const h = Math.floor(sec / 3600);
    const m = Math.floor((sec % 3600) / 60);
    const s = sec % 60;
    return `${pad(h)}:${pad(m)}:${pad(s)}`;
}

function escapeHtml(str) {
    return String(str)
        .replaceAll("&", "&amp;")
        .replaceAll("<", "&lt;")
        .replaceAll(">", "&gt;")
        .replaceAll('"', "&quot;")
        .replaceAll("'", "&#039;");
}

function requestFullscreen() {
    const el = document.documentElement;
    if (el.requestFullscreen) el.requestFullscreen().catch(() => { });
}

async function addViolation(reason) {
    violations += 1;

    // notify server
    if (attemptToken) {
        try {
            await apiPost("./api/heartbeat.php", { token: attemptToken, violation: 1, reason });
        }
        catch { /* ignore */ }
    }

    alert(`Warning: You left the exam screen (${reason}). Violations: ${violations}/${MAX_VIOLATIONS}`);

    if (violations >= MAX_VIOLATIONS) {
        alert("Too many violations. Exam will be submitted now.");
        submitExam(false);
    }
}

// best-effort: detect tab switching & focus loss
document.addEventListener("visibilitychange", () => {
    if (!started || submitted) return;
    if (document.hidden) addViolation("tab hidden / switched");
});

window.addEventListener("blur", () => {
    if (!started || submitted) return;
    addViolation("window lost focus");
});

// fullscreen exit detection (best-effort)
document.addEventListener("fullscreenchange", () => {
    if (!started || submitted) return;
    if (!document.fullscreenElement) {
        addViolation("exited fullscreen");
    }
});

function renderMCQs() {
    mcqWrap.innerHTML = "";
    mcqs.forEach((item, idx) => {
        const card = document.createElement("div");
        card.className = "mcq";
        card.innerHTML = `
      <div class="qhead">
        <div><span class="qnum">Q${idx + 1}.</span> ${escapeHtml(item.q)}</div>
        <span class="badge">1 mark</span>
      </div>
      <div class="options">
        ${item.a.map((opt, i) => `
          <label class="opt">
            <input type="radio" name="q_${idx}" value="${i}" />
            <span>${escapeHtml(opt)}</span>
          </label>
        `).join("")}
      </div>
    `;
        mcqWrap.appendChild(card);
    });
}

function renderTheory() {
    theoryWrap.innerHTML = "";
    theory.forEach((t, idx) => {
        const div = document.createElement("div");
        div.className = "mcq";
        div.innerHTML = `
      <div class="qhead">
        <div><span class="qnum">${escapeHtml(t.title)}</span></div>
        <span class="badge">Manual</span>
      </div>
      <p class="muted" style="margin:8px 0 0; white-space:pre-wrap;">${escapeHtml(t.prompt)}</p>
      <p class="muted" style="margin:8px 0 0;"><em>${escapeHtml(t.hint)}</em></p>
      <label style="margin-top:10px;">
        Answer:
        <textarea id="theory_${idx}" placeholder="Type your answer/code here..."></textarea>
      </label>
    `;
        theoryWrap.appendChild(div);
    });
}

function getObjectiveScore() {
    let score = 0;
    mcqs.forEach((item, idx) => {
        const chosen = document.querySelector(`input[name="q_${idx}"]:checked`);
        if (!chosen) return;
        if (Number(chosen.value) === item.correct) score += 1;
    });
    return score;
}

function updateLiveScore() {
    liveScoreEl.textContent = getObjectiveScore();
}

function startTimer() {
    timerEl.textContent = formatTime(remaining);
    timer = setInterval(() => {
        remaining -= 1;
        timerEl.textContent = formatTime(remaining);

        if (remaining <= 0) {
            clearInterval(timer);
            timer = null;
            submitExam(true);
        }
    }, 1000);
}

function lockInputs() {
    // disable MCQ + theory inputs
    document.querySelectorAll("#examArea input[type=radio], #examArea textarea").forEach(el => el.disabled = true);
}

function buildReview() {
    reviewArea.innerHTML = "<h3>Answer Review (Objective)</h3>";
    mcqs.forEach((item, idx) => {
        const chosen = document.querySelector(`input[name="q_${idx}"]:checked`);
        const chosenIdx = chosen ? Number(chosen.value) : null;

        const isCorrect = chosenIdx === item.correct;
        const chosenText = chosenIdx === null ? "Not answered" : item.a[chosenIdx];
        const correctText = item.a[item.correct];

        const div = document.createElement("div");
        div.className = "mcq";
        div.innerHTML = `
      <div class="qhead">
        <div><span class="qnum">Q${idx + 1}.</span> ${escapeHtml(item.q)}</div>
        <span class="badge ${isCorrect ? "correct" : "wrong"}">
          ${chosenIdx === null ? "No answer" : (isCorrect ? "Correct" : "Wrong")}
        </span>
      </div>
      <p style="margin:8px 0 0;">
        Your answer: <strong>${escapeHtml(chosenText)}</strong>
      </p>
      <p style="margin:6px 0 0;">
        Correct answer: <strong class="correct">${escapeHtml(correctText)}</strong>
      </p>
    `;
        reviewArea.appendChild(div);
    });
}

function updateTotal() {
    const obj = Number(rObj.textContent || "0");
    let man = Number(manualScore.value || "0");
    if (Number.isNaN(man)) man = 0;
    if (man < 0) man = 0;
    if (man > 60) man = 60;
    manualScore.value = String(man);

    rTotal.textContent = String(obj + man);
}

async function startAutosaveTheory() {
    if (theoryAutosaveTimer) clearInterval(theoryAutosaveTimer);

    theoryAutosaveTimer = setInterval(async () => {
        if (!attemptToken || submitted) return;
        for (let i = 0; i < theory.length; i++) {
            const ta = document.getElementById(`theory_${i}`);
            if (!ta) continue;

            try {
                await apiPost("./api/save_answer.php", {
                    token: attemptToken,
                    qtype: "theory",
                    qindex: i,
                    answer: ta.value
                });
            } catch {
                // ignore (network issue). student can continue.
            }
        }
    }, THEORY_AUTOSAVE_MS);
}

async function startHeartbeat() {
    if (heartbeatTimer) clearInterval(heartbeatTimer);

    heartbeatTimer = setInterval(async () => {
        if (!attemptToken || submitted) return;
        try {
            await apiPost("./api/heartbeat.php", { token: attemptToken, violation: 0 });
        } catch {
            // ignore
        }
    }, HEARTBEAT_MS);
}

async function submitExam(auto = false) {
    if (submitted) return;
    submitted = true;

    if (timer) { clearInterval(timer); timer = null; }
    if (theoryAutosaveTimer) { clearInterval(theoryAutosaveTimer); theoryAutosaveTimer = null; }
    if (heartbeatTimer) { clearInterval(heartbeatTimer); heartbeatTimer = null; }

    const name = document.getElementById("studentName").value.trim();
    if (!name) {
        alert("Please enter candidate full name before submitting.");
        submitted = false;
        return;
    }

    const sid = document.getElementById("studentId").value.trim();
    const objScore = getObjectiveScore();

    lockInputs();

    examArea.classList.add("hidden");
    resultArea.classList.remove("hidden");

    rName.textContent = name;
    rId.textContent = sid || "-";
    rObj.textContent = objScore;

    buildReview();

    if (auto) alert("Time up! Exam auto-submitted.");

    // notify server of submission
    if (attemptToken) {
        try {
            await apiPost("./api/submit.php", { token: attemptToken, objectiveScore: objScore });
        } catch {
            // ignore
        }
    }

    updateTotal();
}

// ----- Events -----
startBtn.addEventListener("click", async () => {
    const name = document.getElementById("studentName").value.trim();
    if (!name) return alert("Enter the candidate full name first.");

    if (started) return;
    started = true;

    // start attempt on server
    const sid = document.getElementById("studentId").value.trim();
    try {
        const resp = await apiPost("./api/start.php", { name, studentId: sid });
        if (!resp.token) {
            started = false;
            return alert("Could not start exam. Check server /api/start.php");
        }
        attemptToken = resp.token;
    } catch (e) {
        started = false;
        return alert("Server error starting exam. Ensure hosting + PHP API is working.");
    }

    // UI render
    renderMCQs();
    renderTheory();
    examArea.classList.remove("hidden");
    startBtn.disabled = true;

    // fullscreen best-effort
    requestFullscreen();

    // MCQ real-time save + live score
    mcqWrap.addEventListener("change", async (e) => {
        updateLiveScore();

        const input = e.target;
        if (!input || !input.name || !input.name.startsWith("q_")) return;
        const qindex = Number(input.name.split("_")[1]);

        if (!attemptToken || submitted) return;

        try {
            await apiPost("./api/save_answer.php", {
                token: attemptToken,
                qtype: "mcq",
                qindex,
                answer: input.value
            });
        } catch {
            // ignore network errors
        }
    });

    // autosave theory + heartbeat
    startAutosaveTheory();
    startHeartbeat();

    startTimer();
    updateLiveScore();
});

submitBtn.addEventListener("click", () => {
    if (confirm("Submit exam now?")) submitExam(false);
});
submitBtn2.addEventListener("click", () => {
    if (confirm("Submit exam now?")) submitExam(false);
});

toggleReviewBtn.addEventListener("click", () => {
    reviewArea.classList.toggle("hidden");
});

manualScore.addEventListener("input", updateTotal);

printBtn.addEventListener("click", () => window.print());

resetBtn.addEventListener("click", () => {
    if (!confirm("Reset exam? This clears everything.")) return;
    location.reload();
});

// show initial timer
timerEl.textContent = formatTime(remaining);
